# 编译器和解释器
:label:`sec_hybridize`

目前为止，本书主要关注的是*命令式编程*（imperative programming）。
命令式编程使用诸如`print`、“`+`”和`if`之类的语句来更改程序的状态。
考虑下面这段简单的命令式程序：



```python
def add(a, b):
    return a + b

def fancy_func(a, b, c, d):
    e = add(a, b)
    f = add(c, d)
    g = add(e, f)
    return g

print(fancy_func(1, 2, 3, 4))
```

Python是一种*解释型语言*（interpreted language）。因此，当对上面的`fancy_func`函数求值时，它按顺序执行函数体的操作。也就是说，它将通过对`e = add(a, b)`求值，并将结果存储为变量`e`，从而更改程序的状态。接下来的两个语句`f = add(c, d)`和`g = add(e, f)`也将执行类似地操作，即执行加法计算并将结果存储为变量。 :numref:`fig_compute_graph`说明了数据流。

![命令式编程中的数据流](../img/computegraph.svg)
:label:`fig_compute_graph`

尽管命令式编程很方便，但可能效率不高。一方面原因，Python会单独执行这三个函数的调用，而没有考虑`add`函数在`fancy_func`中被重复调用。如果在一个GPU（甚至多个GPU）上执行这些命令，那么Python解释器产生的开销可能会非常大。此外，它需要保存`e`和`f`的变量值，直到`fancy_func`中的所有语句都执行完毕。这是因为程序不知道在执行语句`e = add(a, b)`和`f = add(c, d)`之后，其他部分是否会使用变量`e`和`f`。

## 符号式编程

考虑另一种选择*符号式编程*（symbolic programming），即代码通常只在完全定义了过程之后才执行计算。这个策略被多个深度学习框架使用，包括Theano和TensorFlow（后者已经获得了命令式编程的扩展）。一般包括以下步骤：

1. 定义计算流程。
1. 将流程编译成可执行的程序。
1. 给定输入，调用编译好的程序执行。

这将允许进行大量的优化。首先，在大多数情况下，我们可以跳过Python解释器。从而消除因为多个更快的GPU与单个CPU上的单个Python线程搭配使用时产生的性能瓶颈。其次，编译器可以将上述代码优化和重写为`print((1 + 2) + (3 + 4))`甚至`print(10)`。因为编译器在将其转换为机器指令之前可以看到完整的代码，所以这种优化是可以实现的。例如，只要某个变量不再需要，编译器就可以释放内存（或者从不分配内存），或者将代码转换为一个完全等价的片段。下面，我们将通过模拟命令式编程来进一步了解符号式编程的概念。



```python
def add_():
    return '''
def add(a, b):
    return a + b
'''

def fancy_func_():
    return '''
def fancy_func(a, b, c, d):
    e = add(a, b)
    f = add(c, d)
    g = add(e, f)
    return g
'''

def evoke_():
    return add_() + fancy_func_() + 'print(fancy_func(1, 2, 3, 4))'

prog = evoke_()
print(prog)
y = compile(prog, '', 'exec')
exec(y)
```

命令式（解释型）编程和符号式编程的区别如下：

* 命令式编程更容易使用。在Python中，命令式编程的大部分代码都是简单易懂的。命令式编程也更容易调试，这是因为无论是获取和打印所有的中间变量值，或者使用Python的内置调试工具都更加简单。
* 符号式编程运行效率更高，更易于移植。符号式编程更容易在编译期间优化代码，同时还能够将程序移植到与Python无关的格式中，从而允许程序在非Python环境中运行，避免了任何潜在的与Python解释器相关的性能问题。

## 混合式编程

历史上，大部分深度学习框架都在命令式编程与符号式编程之间进行选择。例如，Theano、TensorFlow（灵感来自前者）、Keras、CNTK和飞桨采用了符号式编程。相反地，Chainer和PyTorch采取了命令式编程。在后来的版本更新中，TensorFlow2.0、Keras和飞桨增加了命令式编程。


如上所述，飞桨是基于命令式编程并且使用动态计算图。为了能够利用符号式编程的可移植性和效率，开发人员思考能否将这两种编程模型的优点结合起来，于是就产生了飞桨2.0版本。飞桨2.0及以上版本允许用户使用纯命令式编程进行开发和调试，同时能够一行代码转换为符号式程序，以便在需要产品级计算性能和部署时使用。


## `Sequential`的混合式编程

要了解混合式编程的工作原理，最简单的方法是考虑具有多层的深层网络。按照惯例，Python解释器需要执行所有层的代码来生成一条指令，然后将该指令转发到CPU或GPU。对于单个的（快速的）计算设备，这不会导致任何重大问题。另一方面，如果我们使用先进的8-GPU服务器，比如AWS P3dn.24xlarge实例，Python将很难让所有的GPU都保持忙碌。如何解决这个瓶颈，我们可以到后面的多GPU训练和多GPU的简洁实现两节寻找答案。现在，首先，我们定义一个简单的多层感知机。



```python
import paddle
from paddle import nn
from paddle.jit import to_static
from paddle.static import InputSpec
from d2l import paddle as d2l


# 生产网络的工厂模式
# 因为飞桨动态图太快了，需要增加更多的线形图来对比，比如42层。
def get_net():
    blocks = [nn.Linear(512, 512) for i in range(42)] + [
        nn.Linear(512, 256),
        nn.ReLU(),
        nn.Linear(256, 128),
        nn.ReLU(),
        nn.Linear(128, 2)
    ]
    net = nn.Sequential(*blocks)
    return net

x = paddle.randn((1, 512))
net = get_net()
net(x)
```

通过使用`paddle.jit.to_static`函数来转换模型，我们就有能力编译和优化多层感知机中的计算，而模型的计算结果保持不变。



```python
net = to_static(net)
net(x)
```

我们编写与之前相同的代码，再使用`paddle.jit.to_static`简单地转换模型，当完成这些任务后，网络就将得到优化（我们将在下面对性能进行基准测试）。


### 通过混合式编程加速

为了证明通过编译获得了性能改进，我们比较了混合编程前后执行`net(x)`所需的时间。让我们先定义一个度量时间的类，它在本章中在衡量（和改进）模型性能时将非常有用。



```python
#@save
class Benchmark:
    """用于测量运行时间"""
    def __init__(self, description='Done'):
        self.description = description

    def __enter__(self):
        self.timer = d2l.Timer()
        return self

    def __exit__(self, *args):
        print(f'{self.description}: {self.timer.stop():.4f} sec')
```

现在我们可以调用网络两次，一次使用动态图命令式编程，一次使用静态图符号式编程。



```python
net = get_net()
with Benchmark('Paddle动态图命令式编程'):
    for i in range(1000): net(x)

x_spec = InputSpec(shape=[-1, 512], name='x') 
net = to_static(get_net(),input_spec=[x_spec])
with Benchmark('Paddle静态图符号式编程'):
    for i in range(1000): net(x)
```

如以上结果所示，在`nn.Sequential`的实例被函数`paddle.jit.to_static`脚本化后，通过使用符号式编程提高了计算性能。事实上飞桨非常巧妙的实现了动静自然统一，完备实现了一键式动静转换，也就是只需要一条命令，就可以实现动静转换，具体见附录部分。


### 序列化


编译模型的好处之一是我们可以将模型及其参数序列化（保存）到磁盘。这允许这些训练好的模型部署到其他设备上，并且还能方便地使用其他前端编程语言。同时，通常编译模型的代码执行速度也比命令式编程更快。让我们看看`paddle.jit.save`的实际功能。



```python
paddle.jit.save(net, './my_net')
!ls -lh my_net*
```

## 小结

* 命令式编程使得新模型的设计变得容易，因为可以依据控制流编写代码，并拥有相对成熟的Python软件生态。
* 符号式编程要求我们先定义并且编译程序，然后再执行程序，其好处是提高了计算性能。


## 练习


1. 回顾前几章中你感兴趣的模型，你能提高它们的计算性能吗？


## 附录：一键动静转换
动态图和静态图可以写成一份代码，唯一不同的地方就是使用`paddle.enable_static()`切换到静态图，使用`paddle.disable_static()`切换到动态图。飞桨默认是动态图，所以`paddle.disable_static()`这句话是可以省略的。需要注意，在实验中切换到静态图后，若再执行动态图代码（整个《动手学深度学习》的几乎其它所有项目和代码都是动态图代码），不要忘记用`paddle.disable_static()`切换回动态图。

为了更好的表现动静统一，动静两者代码保持一模一样，我们单独写了一个ToArray类，以完成数据转换为ndarray float32类型的任务。


```python
import numpy as np
from paddle.vision.transforms import BaseTransform
class ToArray(BaseTransform):
    """Convert a ``PIL.Image`` to ``numpy.ndarray`` 
    """
    def __init__(self, keys=None):
        super(ToArray, self).__init__(keys)
        
    def _apply_image(self, img):
        """
        Args:
            img (PIL.Image|np.ndarray): Image to be converted to numpy.ndarray.

        Returns:
            numpy.ndarray: Converted image.
        """
        #转换为ndarray类型，并调整uint8类型到float32类型
        out = np.array(img, dtype='float32')
        return out
```

### 静态图
在AIStudio的Tesla V100显卡环境下，我们看到训练速度为`3ms/step`，整个项目执行耗时9.7秒。
如果是第一次执行，会因为数据集下载而多消耗时间，只要再重新运行一次即可。


```python
import paddle
from paddle.vision.transforms import BaseTransform, Compose, Transpose
from paddle.static import InputSpec

# 一句话切换动态图和静态图，默认是动态图模式，也可以用命令paddle.disable_static()显式用打开动态图模式。
paddle.enable_static()
# paddle.disable_static() 

transforms = Compose([Transpose(), ToArray()])
train_dataset = paddle.vision.datasets.MNIST(mode='train', transform=transforms)
test_dataset = paddle.vision.datasets.MNIST(mode='test', transform=transforms)


# 静态图需要指定输入参数的张量信息
x_spec = InputSpec(shape=[-1, 1, 28, 28], dtype='float32', name='x') 
y_spec = InputSpec(shape=[-1, 1], dtype='int64', name='y')
# 直接调用飞桨即成的lenet模型，Model包含了训练功能
lenet = paddle.vision.models.LeNet()
model = paddle.Model(lenet,inputs=x_spec, labels=y_spec)

# 设置训练模型所需的optimizer, loss, metric
model.prepare(
    paddle.optimizer.Adam(learning_rate=0.001, parameters=model.parameters()),
    paddle.nn.CrossEntropyLoss(),
    paddle.metric.Accuracy(topk=(1, 2))
    )

def train():
    # 启动训练
    model.fit(train_dataset, epochs=1, batch_size=32, log_freq=400)

    # 启动评估
    model.evaluate(test_dataset, log_freq=40, batch_size=64)
    
if __name__ == '__main__':
    train()

paddle.disable_static() 
```

### 动态图
在AIStudio的Tesla V100显卡环境下，我们看到训练速度为`6ms/step`，整个项目执行耗时17.1秒。可见静态图对速度的加成还是很可观的，对比动态图加速效果显著！

尽管可以一键转换动态图和静态图，非常方便，但是我们仔细看一下代码，会发现跟平时写的动态图代码有很大的不同，比平常的动态图代码写起来要麻烦一些。一键转换为静态图，提高了运行速度，但是相应的也提高了写代码的难度。代码难易排序为： 纯静态图代码>动静统一代码>纯动态图代码，代码运行速度从快到慢为：纯静态图代码>=动静统一代码>纯动态图代码。



```python
import paddle
from paddle.vision.transforms import BaseTransform, Compose, Transpose
from paddle.static import InputSpec

# 一句话切换动态图和静态图，默认是动态图模式，也可以用命令paddle.disable_static()显式用打开动态图模式。
# paddle.enable_static()
paddle.disable_static() 

transforms = Compose([Transpose(), ToArray()])
train_dataset = paddle.vision.datasets.MNIST(mode='train', transform=transforms)
test_dataset = paddle.vision.datasets.MNIST(mode='test', transform=transforms)


# 静态图需要指定输入参数的张量信息
x_spec = InputSpec(shape=[-1, 1, 28, 28], dtype='float32', name='x') 
y_spec = InputSpec(shape=[-1, 1], dtype='int64', name='y')
# 直接调用飞桨即成的lenet模型，Model包含了训练功能
lenet = paddle.vision.models.LeNet()
model = paddle.Model(lenet,inputs=x_spec, labels=y_spec)

# 设置训练模型所需的optimizer, loss, metric
model.prepare(
    paddle.optimizer.Adam(learning_rate=0.001, parameters=model.parameters()),
    paddle.nn.CrossEntropyLoss(),
    paddle.metric.Accuracy(topk=(1, 2))
    )

def train():
    # 启动训练
    model.fit(train_dataset, epochs=1, batch_size=32, log_freq=400)

    # 启动评估
    model.evaluate(test_dataset, log_freq=40, batch_size=64)
    
if __name__ == '__main__':
    train()
```

[Discussions](https://discuss.d2l.ai/t/2788)

